<?php
/*
 *  This file is part of open3A.

 *  open3A is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.

 *  open3A is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *  2007 - 2020, open3A GmbH - Support@open3A.de
 */
class Kappendix extends PersistentObject {
	public function __construct($id){
		parent::__construct($id);

		$this->customize();
	}
	
	public function checkIBAN($IBAN){
		$I = new IBAN($IBAN);
		return $I->Verify();
	}

	public function getSEPAData(){
		$sepaData = new stdClass();
		$sepaData->IBAN = $this->A("KappendixIBAN");
		$sepaData->BIC = $this->A("KappendixSWIFTBIC");
		$sepaData->MandateDate = $this->A("KappendixIBANMandatDatum");
		$sepaData->MandateID = $this->A("KappendixIBANMandatReferenz") != "" ? $this->A("KappendixIBANMandatReferenz") : substr($this->A("kundennummer").str_replace(" ", "", $this->A("KappendixIBAN")), 0, 34);
		$sepaData->Typ = "";
		
		Aspect::joinPoint("alter", $this, __METHOD__, array($sepaData));
		
		return $sepaData;
	}
	
	public function newAttributes(){
		$A = parent::newAttributes();
		$A->KappendixSameKontoinhaber = 1;
		$A->KappendixErloeskonto = "0";
		
		if($this->customizer != null)
			$this->customizer->customizeNewAttributes($this->getClearClass(get_class($this)), $A);

		if(Session::isPluginLoaded("mArtikelRG")){
			$A->KappendixRabattgruppe = mUserdata::getUDValueS("DefaultValueKappendixKappendixRabattgruppe", 0);
		}
		
		return $A;
	}

	public static function mwst(Stammdaten $S, $Artikel, Adresse $Adresse, $UStIDNr = ""){
		// <editor-fold defaultstate="collapsed" desc="Aspect:jP">
		try {
			$MArgs = func_get_args();
			return Aspect::joinPoint("around", __CLASS__, __METHOD__, $MArgs);
		} catch (AOPNoAdviceException $e) {}
		Aspect::joinPoint("before", __CLASS__, __METHOD__, $MArgs);
		// </editor-fold>
		
		if($Artikel == null)
			return 0;

		$UStIDNr = trim($UStIDNr);
		
		if($Adresse->A("land") == "")
			return $Artikel->A("mwst");
		
		if(!EUCountries::usesVATNumer($S->A("land"))) //Schweiz!
			return $Artikel->A("mwst");
		
		$ISO3166_2 = ISO3166::getCodeToCountry($Adresse->A("land"));
		
		if($S->A("land") == "DE" AND $ISO3166_2 == "NO")
			return 0;
		
		#if(trim($UStIDNr) == "")
		#	return $Artikel->A("mwst");
		
		if(
			$S->A("ustidnr") != ""
			AND $UStIDNr != ""
			AND preg_match("/(^[A-Za-z]{2})/", $S->A("ustidnr"))
			AND preg_match("/(^[A-Za-z]{2})/", $UStIDNr)
			AND strtolower(substr($S->A("ustidnr"), 0, 2)) == strtolower(substr($UStIDNr, 0, 2))) //same country
			return $Artikel->A("mwst");
		
		
		
		if(
			$UStIDNr != "" 
			AND preg_match("/(^[A-Za-z]{2})/", $UStIDNr) 
			AND preg_match("/(^[A-Za-z]{2})/", $S->A("ustidnr"))
			AND strtolower(substr($S->A("ustidnr"), 0, 2)) != strtolower(substr($UStIDNr, 0, 2)) //different country
			AND EUCountries::usesVATNumer($ISO3166_2))
			return 0;
		
		if($UStIDNr == "" AND !EUCountries::usesVATNumer($ISO3166_2))
			return 0;
		
		
		#if($Adresse->A("land") != "" AND $Adresse->A("land") != ISO3166::getCountryToCode($S->A("land")))
		#	return 0;
		
		return $Artikel->A("mwst");
	}
	
	private function updateErloeskonto(){
		$EK = mUserdata::getGlobalSettingValue("DVKappendixErloeskonto", "8400");
		if(mUserdata::getGlobalSettingValue("DVKappendixKundenKonto", "0"))
			$EK += $this->A("kundennummer");
		
		$this->changeA("KappendixErloeskonto", $EK);
	}
	
	private function updateSollkonto(){
		$EK = mUserdata::getGlobalSettingValue("DVKappendixSollkonto", "");
		if(mUserdata::getGlobalSettingValue("DVKappendixKundenKontoSoll", "0"))
			$EK += $this->A("kundennummer");
		
		$this->changeA("KappendixSollkonto", $EK);
	}
	
	public function saveMe($checkUserData = true, $output = false){
		// <editor-fold defaultstate="collapsed" desc="Aspect:jP">
		try {
			$MArgs = func_get_args();
			return Aspect::joinPoint("around", $this, __METHOD__, $MArgs);
		} catch (AOPNoAdviceException $e) {}
		Aspect::joinPoint("before", $this, __METHOD__, $MArgs);
		// </editor-fold>

		if($this->A("KappendixErloeskonto") == "0")
			$this->updateErloeskonto();
		
		if($this->A("KappendixSollkonto") == "")
			$this->updateSollkonto();
				
		$r = parent::saveMe($checkUserData, false);
		
		if(Session::isPluginLoaded("mKlickTipp"))
			KlickTipp::update(new Adresse($this->A("AdresseID")));
		
		// <editor-fold defaultstate="collapsed" desc="Aspect:jP">
		Aspect::joinPoint("after", $this, __METHOD__, $r);
		
		if($output)
			Red::messageSaved();
		
		return $r;
		// </editor-fold>
	}
	
	public function newMe($checkUserData = true, $output = false) {
		if($this->A("KappendixErloeskonto") == "0")
			$this->updateErloeskonto();
		
		if($this->A("KappendixSollkonto") == "")
			$this->updateSollkonto();
		
		return parent::newMe($checkUserData, $output);
	}

	// <editor-fold defaultstate="collapsed" desc="getAdresseIDToKundennummer">
	public static function getAdresseIDToKundennummer($kundennummer){
		$ac = new anyC();
		$ac->setCollectionOf("Kappendix");
		$ac->addAssocV3("kundennummer", "=", $kundennummer);
		$ac->addJoinV3("Adresse", "AdresseID", "=", "AdresseID");

		$E = $ac->getNextEntry();
		if($E == null) return null;

		return $E->A("AdresseID");
	}
	// </editor-fold>
	
	// <editor-fold defaultstate="collapsed" desc="getKappendixToAdresse">
	public static function getKappendixToAdresse($AdresseID){
		$ac = anyC::get("Kappendix");
		$ac->addAssocV3("AdresseID", "=", $AdresseID);

		return $ac->getNextEntry();
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="getKappendixToKundennummer">
	/**
	 * @param string $kundennummer
	 * @return Kappendix
	 */
	public static function getKappendixToKundennummer($kundennummer){
		$ac = new anyC();
		$ac->setCollectionOf("Kappendix");
		$ac->addAssocV3("kundennummer", "=", $kundennummer);

		return $ac->getNextEntry();
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="getKappendixIDToAdresse">
	public static function getKappendixIDToAdresse($AdresseID, $returnKundennummer = false){
		$K = self::getKappendixToAdresse($AdresseID);
		
		if($K != null) {
			if($returnKundennummer) 
				return $K->A("kundennummer");
			
			return $K->getID();
		}
		
		return null;
	}
	// </editor-fold>
}
?>
