<?php
	/**
	 * Bezahlcode 0.3
	 *
	 * Generates SEPA QR-Codes for money transfer
	 *
	 * PHP version tested up to 7.3
	 *
	 * Bezahlcode is distributed under LGPL 3
	 * Copyright (C) 2019 Matthias Schaffer
	 *
	 * @package    Bezahlcode
	 * @author     Matthias Schaffer <hello <at> matthiasschaffer <dot> com>
	 * @copyright  2019
	 * @license    https://www.gnu.org/licenses/lgpl LGPL 3
	 * @link       https://github.com/fellwell5/bezahlcode/
	 * @since      File available since beginning of project
	*/
	class Bezahlcode {
                public $base64_frame = "data:image/png;base64,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 ";
		public $phpqrcode_path = "./phpqrcode.php";
		private $qrprovider, $use_frame, $iban, $bic, $name, $payload, $usage, $amount;
		
		/**
		 * __construct
		 * 
		 * Create class object and save data for further use.
		 *
		 * @param string $iban
		 *
		 * @param string $bic
		 *
		 * @param string $name Name of the bank account owner.
		 *
		 * @param string $qrprovider (optional)
		 * 		Defaults to 'phpqrcode'.
		 * 		options are 'phpqrcode' or 'google'
		 *
		 * @param boolean $use_frame (optional) Use Bezahlcode-Frame around the QRcode. Defined by the public variable $base64_frame
		 *
		 * @access public
		 * @since Method available since Release 0.1
		 */
		function __construct($iban, $bic, $name, $qrprovider = "phpqrcode", $use_frame = true){
			$this->iban = $iban;
			$this->bic = $bic;
			$this->name = $name;
			$this->qrprovider = $qrprovider;
			$this->use_frame = $use_frame;
			
			if ($qrprovider == "phpqrcode") {
				if (!file_exists($this->phpqrcode_path)) {
					die("PHPQRCODE was selected as QR-Code provider, but phpqrcode.php was not found at the specified path. Please set the variable to the path.");
				}
				require_once $this->phpqrcode_path;
				if (!class_exists("QRcode")) {
					die("PHPQRCODE was selected as QR-Code provider, but QRcode class could not be loaded.");
				}
			}
		}
		
		/**
		 * generatePayload
		 * 
		 * Generate payload array with specified data and save it for further use.
		 *
		 * @param string $usage
		 *
		 * @param decimal $amount Formatted like 10.56
		 *
		 * @access public
		 * @since Method available since Release 0.1
		 */
		public function generatePayload($usage, $amount){
			$this->usage = $usage;
			$this->amount = $amount;
			$payload = array("BCD", "001", "1", "SCT", $this->bic, $this->name, $this->iban, "EUR".$amount, "", "", $usage);
			$this->payload = implode("\n", $payload);
		}
		
		/**
		 * getPayload
		 * 
		 * Return the payload saved from Bezahlcode->generatePayload().
		 *
		 * @return string
		 *
		 * @access private
		 * @since Method available since Release 0.1
		 */
		private function getPayload(){
			return $this->payload;
		}
		
		/**
		 * getQRCode
		 * 
		 * Return image resource with specified payload 
		 *
		 * @param string $data Should be the payload but any text could be used.
		 *
		 * @return resource
		 *
		 * @access private
		 * @since Method available since Release 0.2
		 */
		private function getQRCode(){
			if ($this->qrprovider == "phpqrcode") {
				ob_start();
				QRcode::png($this->payload, false, QR_ECLEVEL_L, 9, 1);
				$qr = ob_get_contents();
				ob_end_clean();
				$qr = imagecreatefromstring($qr);
			} elseif ($this->qrprovider == "google") {
				$qr = imagecreatefrompng("https://chart.apis.google.com/chart?cht=qr&chs=350x350&chld=L|0&chl=".urlencode($this->payload));
			} else {
				die("QRCode provider $this->qrprovider is not valid");
			}
			
			if ($this->use_frame) {
				$img = str_replace('data:image/png;base64,', '', $this->base64_frame);
				$img = str_replace(' ', '+', $img);
				$img = base64_decode($img);
				$layer = imagecreatefromstring($img);
				imagealphablending($layer, true);
				imagesavealpha($layer, true);
				imagecopy($layer, $qr, 33, 33, 0, 0, 350, 350);
				
				return $layer;
			} else {
				return $qr;
			}
			
		}
			
    
		/**
		 * selectType
		 * 
		 * Return the content type and extension from specified image type.
		 *
		 * @param string $type optional The file type as which the image should be generated.
		 * 		Defaults to '' if nothing is specified jpg is used. See at Bezahlcode->selectType.
		 *
		 * @return array With the content type and the extension.
		 * 		for example: array("image/png", "png");
		 *
		 * @access private
		 * @since Method available since Release 0.2
		 */
		private function selectType($type=''){
			switch(strtolower($type)){
				case "gif": return ["image/gif", "gif"];
				case "png": return ["image/png", "png"];
				default: 		return ["image/jpeg", "jpg"];
			}
		}
    
		/**
		 * generateBase64
		 * 
		 * Generate a Bezahlcode with saved Payload and return the Bezahlcode as base64 encoded string.
		 *
		 * @param string $type optional The file type as which the image should be generated.
		 * 		Defaults to '' if nothing is specified jpg is used. See at Bezahlcode->selectType.
		 *
		 * @return string The generated base64 string.
		 *
		 * @access public
		 * @since Method available since Release 0.1
		 */
		public function generateBase64($type=''){
			$type = $this->selectType($type);
			$bezahlcode = $this->getQRCode();
			
			ob_start();
			if($type[1] == "gif"){
				imagegif($bezahlcode);
			}elseif($type[1] == "png"){
				imagepng($bezahlcode);
			}else{
				imagejpeg($bezahlcode);
			}
			$imagedata = ob_get_contents();
			ob_end_clean();
			imagedestroy($bezahlcode);
			
			header("Content-Type: text/html; charset=UTF-8");
			return 'data:'.$type[0].';base64,'.base64_encode($imagedata);
		}
		
		/**
		 * saveImage
		 * 
		 * Generate a Bezahlcode with saved Payload and save it as file.
		 *
		 * @param string $filename optional The path and filename where the Bezahlcode should be saved.
		 * 		Defaults to '' and then a filename is generated from the usage field.
		 * @param string $type optional The file type as which the image should be saved.
		 * 		Defaults to '' if nothing is specified jpg is used. See at Bezahlcode->selectType.
		 *
		 * @return string The filename where the Bezahlcode was saved.
		 *
		 * @access public
		 * @since Method available since Release 0.2
		 */
		public function saveImage($filename='',$type=''){
			$type = $this->selectType($type);
			$bezahlcode = $this->getQRCode();
			
			if(empty($filename)) $filename = "bezahlcode_".preg_replace('/[^a-zA-Z0-9\-\_]/','', $this->usage).".".$type[1];
			
			if($type[1] == "gif"){
				imagegif($bezahlcode, $filename);
			}elseif($type[1] == "png"){
				imagepng($bezahlcode, $filename);
			}else{
				imagejpeg($bezahlcode, $filename);
			}
			
			return $filename;
		}
		
		
		/**
		 * outputImage
		 * 
		 * Generate a Bezahlcode and output it to the webbrowser.
		 * After the output the function exit() is called to suppress further output.
		 *
		 * @param string $type optional The file type as which the image should be outputted.
		 * 		Defaults to '' if nothing is specified jpg is used. See at Bezahlcode->selectType.
		 *
		 *
		 * @access public
		 * @since Method available since Release 0.2
		 */
		public function outputImage($type=''){
			$type = $this->selectType($type);
			$bezahlcode = $this->getQRCode();
			
			header("Content-type: ".$type[0]);
			header("Content-Disposition: inline; filename=bezahlcode.".$type[1]);
			if($type[1] == "gif"){
				imagegif($bezahlcode);
			}elseif($type[1] == "png"){
				imagepng($bezahlcode);
			}else{
				imagejpeg($bezahlcode);
			}
			imagedestroy($bezahlcode);
			
			exit();
		}
	}
?>
